'Drop example V1.0
'By David Powell
'http://www.loadcode.co.uk
'Released into the public domain

'Axe graphic from:
'http://openclipart.org/detail/171091/axe-by-hatalar205-171091

'Lemon graphic from:
'http://openclipart.org/detail/189589/lemon-citrina-by-keistutis-189589

'This example uses a lot of contantly rotating masks and mask to mask collision detection.
'While the collision detection is fast, the mask rotation can take a lot of processing time.
'Some targets fair better than others, but all benefit greatly from running in Relase mode rather than Debug.

'Also of note is that the masks were generated from versions of the images that had their
'shadows removed. Therefore a collision will only register for the body of the object
'and not the shadow.

'#FAST_SYNC_PROJECT_DATA = True

Strict

Import mojo
Import brl.pool
Import collisionmask

Public
	'Summary: Start the DropApp application
	Function Main:Int()
		New DropApp()
		Return 0
	End

Public
	'Summary: The Drop application
	Class DropApp Extends App
		Private
			'The axe image and master mask
			'Each axe will store it's own simple Mask that reflects
			'the current scale and rotation.
			Field axeImage:Image
			Field axeMasterMask:MasterMask
			
			'The lemon image and mask
			'The lemon does not rotate or scale, so the master mask
			'is used directly in any collision checks 
			Field lemonImage:Image
			Field lemonMasterMask:MasterMask
			
			'The maximum number of Axes on screen at once
			'Adjusting this number will affect the performance of the application
			Const MaxAxes:Int = 10
			
			'A list of axes currently on screen
			Field axeList:List<Axe>
			
			'The hit flag is set when the lemon is in contact with an axe
			Field hit:Bool = False
	
		Public
			'Summary: When the application is created; everything is initialised
			Method OnCreate:Int()
				'Load the images and masks
				
				'As the images are loaded with the Image.XYPadding flag set, when the
				'masks were generated they had a border of 1 pixel trimmed so that the
				'size is the same as the image after padding removal.
				'If a sprite atlas was used then mask trimming would not be required.
				
				axeImage = LoadImage("monkey://data/axe_shadow.png", 1, Image.XYPadding | Image.MidHandle)
				If axeImage = Null Then Error("Cannot load axe_shadow.png")
				
				axeMasterMask = LoadMask("monkey://data/axe_noshadow.cmsk", Mask.MidHandle)
				If axeMasterMask = Null Then Error("Cannot load axe_noshadow.cmsk")
				
				lemonImage = LoadImage("monkey://data/lemon_shadow.png", 1, Image.XYPadding | Image.MidHandle)
				If lemonImage = Null Then Error("Cannot load lemon_shadow.png")
				
				lemonMasterMask = LoadMask("monkey://data/lemon_noshadow.cmsk", Mask.MidHandle)
				If lemonMasterMask = Null Then Error("Cannot load lemon_noshadow.cmsk")
				
				'The axe pool is initialised now that the axe mask has been loaded successfully
				Axe.InitPool(axeMasterMask, MaxAxes)
				axeList = New List<Axe>
				
				Seed = Millisecs()
				
				SetUpdateRate(60)
				
				Return 0
			End
		
		Public
			'Summary: Update the current state of the application
			Method OnUpdate:Int()
				'Create new axes
				If axeList.Count() < MaxAxes
					If Rnd() < 0.01 'There is a 1% chance of creating a new axe each update
						Local axe:Axe = Axe.Create()
						axe.Init()
						axeList.AddLast(axe)
					EndIf
				EndIf
				
				'Default the hit flag to false
				hit = False
				
				'Go though each axe
				For Local axe:Axe = EachIn axeList
					'If the axe is off the bottom of the screen, then remove it
					If (axe.y > DeviceHeight() +Axe.maxSize)
						axeList.Remove(axe)
						axe.Destroy()
						Continue
					EndIf
					
					'Update the axe position and angle
					axe.y += axe.speed
					axe.angle += axe.speed
					
					'Scale and rotate the master mask based upon the axe's current attributes
					'The resulting mask data is written to the mask associated with this axe
					axeMasterMask.ScaleAndRotate(axe.size, axe.size, axe.angle, axe.mask)
					
					If hit = False ' Only check for collisions if there hasn't been one yet
						'Check to see if this rotated and scaled axe mask collides with the lemon mask
						If axe.mask.CollideMask(axe.x, axe.y, lemonMasterMask, MouseX(), DeviceHeight() -80) = True
							hit = True
						EndIf
					EndIf
				Next
			
				Return 0
			End
		
		Public
			'Summary: Render the current state of the application
			Method OnRender:Int()
				'Choose a background colour based upon the hit flag
				If hit = False
					Cls(128, 128, 128)
				Else
					Cls(255, 128, 128)
				EndIf
				
				'Draw each axe on the screen
				For Local axe:Axe = EachIn axeList
					DrawImage(axeImage, axe.x, axe.y, axe.angle, axe.size, axe.size)
					'axe.mask.DebugDraw(axe.x, axe.y, 0.5)
				Next
				
				'Draw the lemon
				DrawImage(lemonImage, MouseX(), DeviceHeight() -80)
				'lemonMasterMask.DebugDraw(MouseX(),  DeviceHeight() -80, 0.5)
				
				Return 0
			End
	End

Public
	'Summary: The Axe object
	Class Axe
		Public
			'The attributes of an Axe
			Field x:Int
			Field y:Int
			Field size:Float
			Field angle:Int
			Field speed:Int
			Field mask:Mask
			
		Private
			'Global information about all Axes
			'An axe pool is used so that New is not called in the main game loop
			Global axePool:Pool<Axe>
			Global axeMasterMask:MasterMask
			Global maxSize:Int
			
		Public
			'Summary: Initialise the axe pool with Axe objects
			Function InitPool:Void(masterMask:MasterMask, maxAxes:Int)
				axeMasterMask = masterMask
				'The maximum size of an axe (mask) is the length of the diagonal from corner to corner (Pythagorean theorem)
				maxSize = Sqrt(axeMasterMask.Width() * axeMasterMask.Width() +axeMasterMask.Height() * axeMasterMask.Height())
				axePool = New Pool<Axe>(maxAxes)
			End
			
		Public
			'Sumary: Create an Axe
			Function Create:Axe()
				'Axe objects are retrieved from the axe pool
		        Return axePool.Allocate().Init()
		    End
			
		Public
			'Summary: Destroy the Axe
			Method Destroy:Void()
				'Axe objects are returned to the axe pool
		        axePool.Free(Self)
		    End
			
		Public
			'Summary: Initialse an Axe
			Method New()
				'A mask for this axe is pre-allocated from the master mask
				'This mask is then updated each time the mask is rotated so that
				'there is no need to call New in the main game loop
				mask = axeMasterMask.PreAllocateMask(1.0, 1.0)
			End
			
		Private
			'Summary: Initialise the Axe
			Method Init:Axe()
				'The Axe is initialised with random values, except for the Y position
				'which is set so that the Axe starts off the top of the screen, no
				'matter how it is rotated.
				x = Rnd(0, DeviceWidth())
				y = -maxSize
				size = Rnd(0.5, 1.0)
				angle = Rnd(0, 360)
				speed = Rnd(1, 4)
				Return Self
			End
	End
